import crypto from "crypto";
import fs from "fs";
import { Worker } from "next/dist/compiled/jest-worker";
import findUp from "next/dist/compiled/find-up";
import { join as pathJoin, relative, resolve as pathResolve, sep } from "path";
import Watchpack from "next/dist/compiled/watchpack";
import { ampValidation } from "../../build/output";
import { PUBLIC_DIR_MIDDLEWARE_CONFLICT } from "../../lib/constants";
import { fileExists } from "../../lib/file-exists";
import { findPagesDir } from "../../lib/find-pages-dir";
import loadCustomRoutes from "../../lib/load-custom-routes";
import { verifyTypeScriptSetup } from "../../lib/verifyTypeScriptSetup";
import { verifyPartytownSetup } from "../../lib/verify-partytown-setup";
import { PHASE_DEVELOPMENT_SERVER, CLIENT_STATIC_FILES_PATH, DEV_CLIENT_PAGES_MANIFEST, DEV_MIDDLEWARE_MANIFEST, COMPILER_NAMES } from "../../shared/lib/constants";
import Server, { WrappedBuildError } from "../next-server";
import { getRouteMatcher } from "../../shared/lib/router/utils/route-matcher";
import { getMiddlewareRouteMatcher } from "../../shared/lib/router/utils/middleware-route-matcher";
import { normalizePagePath } from "../../shared/lib/page-path/normalize-page-path";
import { absolutePathToPage } from "../../shared/lib/page-path/absolute-path-to-page";
import Router from "../router";
import { getPathMatch } from "../../shared/lib/router/utils/path-match";
import { pathHasPrefix } from "../../shared/lib/router/utils/path-has-prefix";
import { removePathPrefix } from "../../shared/lib/router/utils/remove-path-prefix";
import { eventCliSession } from "../../telemetry/events";
import { Telemetry } from "../../telemetry/storage";
import { setGlobal } from "../../trace";
import HotReloader from "./hot-reloader";
import { findPageFile, isLayoutsLeafPage } from "../lib/find-page-file";
import { getNodeOptionsWithoutInspect } from "../lib/utils";
import { withCoalescedInvoke } from "../../lib/coalesced-function";
import { loadDefaultErrorComponents } from "../load-components";
import { DecodeError, MiddlewareNotFoundError } from "../../shared/lib/utils";
import { createOriginalStackFrame, getErrorSource, getSourceById, parseStack } from "next/dist/compiled/@next/react-dev-overlay/dist/middleware";
import * as Log from "../../build/output/log";
import isError, { getProperError } from "../../lib/is-error";
import { getRouteRegex } from "../../shared/lib/router/utils/route-regex";
import { getSortedRoutes, isDynamicRoute } from "../../shared/lib/router/utils";
import { runDependingOnPageType } from "../../build/entries";
import { getPageStaticInfo } from "../../build/analysis/get-page-static-info";
import { normalizePathSep } from "../../shared/lib/page-path/normalize-path-sep";
import { normalizeAppPath } from "../../shared/lib/router/utils/app-paths";
import { getPossibleMiddlewareFilenames, isMiddlewareFile, NestedMiddlewareError } from "../../build/utils";
import { getDefineEnv } from "../../build/webpack-config";
import loadJsConfig from "../../build/load-jsconfig";
// Load ReactDevOverlay only when needed
let ReactDevOverlayImpl;
const ReactDevOverlay = (props)=>{
    if (ReactDevOverlayImpl === undefined) {
        ReactDevOverlayImpl = require("next/dist/compiled/@next/react-dev-overlay/dist/client").ReactDevOverlay;
    }
    return ReactDevOverlayImpl(props);
};
export default class DevServer extends Server {
    addedUpgradeListener = false;
    getStaticPathsWorker() {
        if (this.staticPathsWorker) {
            return this.staticPathsWorker;
        }
        this.staticPathsWorker = new Worker(require.resolve("./static-paths-worker"), {
            maxRetries: 1,
            numWorkers: this.nextConfig.experimental.cpus,
            enableWorkerThreads: this.nextConfig.experimental.workerThreads,
            forkOptions: {
                env: {
                    ...process.env,
                    // discard --inspect/--inspect-brk flags from process.env.NODE_OPTIONS. Otherwise multiple Node.js debuggers
                    // would be started if user launch Next.js in debugging mode. The number of debuggers is linked to
                    // the number of workers Next.js tries to launch. The only worker users are interested in debugging
                    // is the main Next.js one
                    NODE_OPTIONS: getNodeOptionsWithoutInspect()
                }
            }
        });
        this.staticPathsWorker.getStdout().pipe(process.stdout);
        this.staticPathsWorker.getStderr().pipe(process.stderr);
        return this.staticPathsWorker;
    }
    constructor(options){
        var ref, ref1;
        super({
            ...options,
            dev: true
        });
        this.renderOpts.dev = true;
        this.renderOpts.ErrorDebug = ReactDevOverlay;
        this.devReady = new Promise((resolve)=>{
            this.setDevReady = resolve;
        });
        this.renderOpts.ampSkipValidation = ((ref = this.nextConfig.experimental) == null ? void 0 : (ref1 = ref.amp) == null ? void 0 : ref1.skipValidation) ?? false;
        this.renderOpts.ampValidator = (html, pathname)=>{
            const validatorPath = this.nextConfig.experimental && this.nextConfig.experimental.amp && this.nextConfig.experimental.amp.validator;
            const AmpHtmlValidator = require("next/dist/compiled/amphtml-validator");
            return AmpHtmlValidator.getInstance(validatorPath).then((validator)=>{
                const result = validator.validateString(html);
                ampValidation(pathname, result.errors.filter((e)=>e.severity === "ERROR").filter((e)=>this._filterAmpDevelopmentScript(html, e)), result.errors.filter((e)=>e.severity !== "ERROR"));
            });
        };
        if (fs.existsSync(pathJoin(this.dir, "static"))) {
            Log.warn(`The static directory has been deprecated in favor of the public directory. https://nextjs.org/docs/messages/static-dir-deprecated`);
        }
        // setup upgrade listener eagerly when we can otherwise
        // it will be done on the first request via req.socket.server
        if (options.httpServer) {
            this.setupWebSocketHandler(options.httpServer);
        }
        this.isCustomServer = !options.isNextDevCommand;
        const { pagesDir , appDir  } = findPagesDir(this.dir, !!this.nextConfig.experimental.appDir);
        this.pagesDir = pagesDir;
        this.appDir = appDir;
    }
    getBuildId() {
        return "development";
    }
    async addExportPathMapRoutes() {
        // Makes `next export` exportPathMap work in development mode.
        // So that the user doesn't have to define a custom server reading the exportPathMap
        if (this.nextConfig.exportPathMap) {
            Log.info("Defining routes from exportPathMap");
            const exportPathMap = await this.nextConfig.exportPathMap({}, {
                dev: true,
                dir: this.dir,
                outDir: null,
                distDir: this.distDir,
                buildId: this.buildId
            }) // In development we can't give a default path mapping
            ;
            for(const path in exportPathMap){
                const { page , query ={}  } = exportPathMap[path];
                this.router.addFsRoute({
                    match: getPathMatch(path),
                    type: "route",
                    name: `${path} exportpathmap route`,
                    fn: async (req, res, _params, parsedUrl)=>{
                        const { query: urlQuery  } = parsedUrl;
                        Object.keys(urlQuery).filter((key)=>query[key] === undefined).forEach((key)=>Log.warn(`Url '${path}' defines a query parameter '${key}' that is missing in exportPathMap`));
                        const mergedQuery = {
                            ...urlQuery,
                            ...query
                        };
                        await this.render(req, res, page, mergedQuery, parsedUrl, true);
                        return {
                            finished: true
                        };
                    }
                });
            }
        }
    }
    async startWatcher() {
        if (this.webpackWatcher) {
            return;
        }
        const regexPageExtension = new RegExp(`\\.+(?:${this.nextConfig.pageExtensions.join("|")})$`);
        let resolved = false;
        return new Promise(async (resolve, reject)=>{
            if (this.pagesDir) {
                // Watchpack doesn't emit an event for an empty directory
                fs.readdir(this.pagesDir, (_, files)=>{
                    if (files == null ? void 0 : files.length) {
                        return;
                    }
                    if (!resolved) {
                        resolve();
                        resolved = true;
                    }
                });
            }
            const pages = this.pagesDir ? [
                this.pagesDir
            ] : [];
            const app = this.appDir ? [
                this.appDir
            ] : [];
            const directories = [
                ...pages,
                ...app
            ];
            const files1 = this.pagesDir ? getPossibleMiddlewareFilenames(pathJoin(this.pagesDir, ".."), this.nextConfig.pageExtensions) : [];
            let nestedMiddleware = [];
            const envFiles = [
                ".env.development.local",
                ".env.local",
                ".env.development",
                ".env", 
            ].map((file)=>pathJoin(this.dir, file));
            files1.push(...envFiles);
            // tsconfig/jsonfig paths hot-reloading
            const tsconfigPaths = [
                pathJoin(this.dir, "tsconfig.json"),
                pathJoin(this.dir, "jsconfig.json"), 
            ];
            files1.push(...tsconfigPaths);
            const wp = this.webpackWatcher = new Watchpack({
                ignored: (pathname)=>{
                    return !files1.some((file)=>file.startsWith(pathname)) && !directories.some((dir)=>pathname.startsWith(dir) || dir.startsWith(pathname));
                }
            });
            wp.watch({
                directories: [
                    this.dir
                ],
                startTime: 0
            });
            const fileWatchTimes = new Map();
            let enabledTypeScript = this.usingTypeScript;
            wp.on("aggregated", async ()=>{
                let middlewareMatchers;
                const routedPages = [];
                const knownFiles = wp.getTimeInfoEntries();
                const appPaths = {};
                const edgeRoutesSet = new Set();
                const pageNameSet = new Set();
                const conflictingAppPagePaths = [];
                let envChange = false;
                let tsconfigChange = false;
                for (const [fileName, meta] of knownFiles){
                    if (!files1.includes(fileName) && !directories.some((dir)=>fileName.startsWith(dir))) {
                        continue;
                    }
                    const watchTime = fileWatchTimes.get(fileName);
                    const watchTimeChange = watchTime && watchTime !== (meta == null ? void 0 : meta.timestamp);
                    fileWatchTimes.set(fileName, meta.timestamp);
                    if (envFiles.includes(fileName)) {
                        if (watchTimeChange) {
                            envChange = true;
                        }
                        continue;
                    }
                    if (tsconfigPaths.includes(fileName)) {
                        if (fileName.endsWith("tsconfig.json")) {
                            enabledTypeScript = true;
                        }
                        if (watchTimeChange) {
                            tsconfigChange = true;
                        }
                        continue;
                    }
                    if ((meta == null ? void 0 : meta.accuracy) === undefined || !regexPageExtension.test(fileName)) {
                        continue;
                    }
                    const isAppPath = Boolean(this.appDir && normalizePathSep(fileName).startsWith(normalizePathSep(this.appDir)));
                    const rootFile = absolutePathToPage(fileName, {
                        pagesDir: this.dir,
                        extensions: this.nextConfig.pageExtensions
                    });
                    const staticInfo = await getPageStaticInfo({
                        pageFilePath: fileName,
                        nextConfig: this.nextConfig,
                        page: rootFile,
                        isDev: true
                    });
                    if (isMiddlewareFile(rootFile)) {
                        var ref;
                        this.actualMiddlewareFile = rootFile;
                        middlewareMatchers = ((ref = staticInfo.middleware) == null ? void 0 : ref.matchers) || [
                            {
                                regexp: ".*"
                            }, 
                        ];
                        continue;
                    }
                    if (fileName.endsWith(".ts") || fileName.endsWith(".tsx")) {
                        enabledTypeScript = true;
                    }
                    let pageName = absolutePathToPage(fileName, {
                        pagesDir: isAppPath ? this.appDir : this.pagesDir,
                        extensions: this.nextConfig.pageExtensions,
                        keepIndex: isAppPath
                    });
                    if (isAppPath) {
                        if (!isLayoutsLeafPage(fileName)) {
                            continue;
                        }
                        const originalPageName = pageName;
                        pageName = normalizeAppPath(pageName) || "/";
                        if (!appPaths[pageName]) {
                            appPaths[pageName] = [];
                        }
                        appPaths[pageName].push(originalPageName);
                        if (routedPages.includes(pageName)) {
                            continue;
                        }
                    } else {
                        // /index is preserved for root folder
                        pageName = pageName.replace(/\/index$/, "") || "/";
                    }
                    if (pageNameSet.has(pageName)) {
                        conflictingAppPagePaths.push(pageName);
                    } else {
                        pageNameSet.add(pageName);
                    }
                    /**
           * If there is a middleware that is not declared in the root we will
           * warn without adding it so it doesn't make its way into the system.
           */ if (/[\\\\/]_middleware$/.test(pageName)) {
                        nestedMiddleware.push(pageName);
                        continue;
                    }
                    await runDependingOnPageType({
                        page: pageName,
                        pageRuntime: staticInfo.runtime,
                        onClient: ()=>{},
                        onServer: ()=>{
                            routedPages.push(pageName);
                        },
                        onEdgeServer: ()=>{
                            routedPages.push(pageName);
                            edgeRoutesSet.add(pageName);
                        }
                    });
                }
                const numConflicting = conflictingAppPagePaths.length;
                if (numConflicting > 0) {
                    Log.error(`Conflicting app and page file${numConflicting === 1 ? " was" : "s were"} found, please remove the conflicting files to continue:`);
                    for (const p of conflictingAppPagePaths){
                        Log.error(`  "pages${p}" - "app${p}"`);
                    }
                //process.exit(1)
                }
                if (!this.usingTypeScript && enabledTypeScript) {
                    // we tolerate the error here as this is best effort
                    // and the manual install command will be shown
                    await this.verifyTypeScript().then(()=>{
                        tsconfigChange = true;
                    }).catch(()=>{});
                }
                if (envChange || tsconfigChange) {
                    var ref2, ref3, ref4;
                    if (envChange) {
                        this.loadEnvConfig({
                            dev: true,
                            forceReload: true
                        });
                    }
                    let tsconfigResult;
                    if (tsconfigChange) {
                        try {
                            tsconfigResult = await loadJsConfig(this.dir, this.nextConfig);
                        } catch (_) {
                        /* do we want to log if there are syntax errors in tsconfig  while editing? */ }
                    }
                    (ref2 = this.hotReloader) == null ? void 0 : (ref3 = ref2.activeConfigs) == null ? void 0 : ref3.forEach((config, idx)=>{
                        const isClient = idx === 0;
                        const isNodeServer = idx === 1;
                        const isEdgeServer = idx === 2;
                        const hasRewrites = this.customRoutes.rewrites.afterFiles.length > 0 || this.customRoutes.rewrites.beforeFiles.length > 0 || this.customRoutes.rewrites.fallback.length > 0;
                        if (tsconfigChange) {
                            var ref14, ref6;
                            (ref14 = config.resolve) == null ? void 0 : (ref6 = ref14.plugins) == null ? void 0 : ref6.forEach((plugin)=>{
                                // look for the JsConfigPathsPlugin and update with
                                // the latest paths/baseUrl config
                                if (plugin && plugin.jsConfigPlugin && tsconfigResult) {
                                    var ref, ref8, ref9;
                                    const { resolvedBaseUrl , jsConfig  } = tsconfigResult;
                                    const currentResolvedBaseUrl = plugin.resolvedBaseUrl;
                                    const resolvedUrlIndex = (ref = config.resolve) == null ? void 0 : (ref8 = ref.modules) == null ? void 0 : ref8.findIndex((item)=>item === currentResolvedBaseUrl);
                                    if (resolvedBaseUrl && resolvedBaseUrl !== currentResolvedBaseUrl) {
                                        var ref10, ref11;
                                        // remove old baseUrl and add new one
                                        if (resolvedUrlIndex && resolvedUrlIndex > -1) {
                                            var ref12, ref13;
                                            (ref12 = config.resolve) == null ? void 0 : (ref13 = ref12.modules) == null ? void 0 : ref13.splice(resolvedUrlIndex, 1);
                                        }
                                        (ref10 = config.resolve) == null ? void 0 : (ref11 = ref10.modules) == null ? void 0 : ref11.push(resolvedBaseUrl);
                                    }
                                    if ((jsConfig == null ? void 0 : (ref9 = jsConfig.compilerOptions) == null ? void 0 : ref9.paths) && resolvedBaseUrl) {
                                        Object.keys(plugin.paths).forEach((key)=>{
                                            delete plugin.paths[key];
                                        });
                                        Object.assign(plugin.paths, jsConfig.compilerOptions.paths);
                                        plugin.resolvedBaseUrl = resolvedBaseUrl;
                                    }
                                }
                            });
                        }
                        if (envChange) {
                            var ref7;
                            (ref7 = config.plugins) == null ? void 0 : ref7.forEach((plugin)=>{
                                // we look for the DefinePlugin definitions so we can
                                // update them on the active compilers
                                if (plugin && typeof plugin.definitions === "object" && plugin.definitions.__NEXT_DEFINE_ENV) {
                                    var ref;
                                    const newDefine = getDefineEnv({
                                        dev: true,
                                        config: this.nextConfig,
                                        distDir: this.distDir,
                                        isClient,
                                        hasRewrites,
                                        hasReactRoot: (ref = this.hotReloader) == null ? void 0 : ref.hasReactRoot,
                                        isNodeServer,
                                        isEdgeServer
                                    });
                                    Object.keys(plugin.definitions).forEach((key)=>{
                                        if (!(key in newDefine)) {
                                            delete plugin.definitions[key];
                                        }
                                    });
                                    Object.assign(plugin.definitions, newDefine);
                                }
                            });
                        }
                    });
                    (ref4 = this.hotReloader) == null ? void 0 : ref4.invalidate();
                }
                if (nestedMiddleware.length > 0) {
                    Log.error(new NestedMiddlewareError(nestedMiddleware, this.dir, this.pagesDir).message);
                    nestedMiddleware = [];
                }
                // Make sure to sort parallel routes to make the result deterministic.
                this.appPathRoutes = Object.fromEntries(Object.entries(appPaths).map(([k, v])=>[
                        k,
                        v.sort()
                    ]));
                const edgeRoutes = Array.from(edgeRoutesSet);
                this.edgeFunctions = getSortedRoutes(edgeRoutes).map((page)=>{
                    const matchedAppPaths = this.getOriginalAppPaths(page);
                    if (Array.isArray(matchedAppPaths)) {
                        page = matchedAppPaths[0];
                    }
                    const edgeRegex = getRouteRegex(page);
                    return {
                        match: getRouteMatcher(edgeRegex),
                        page,
                        re: edgeRegex.re
                    };
                });
                this.middleware = middlewareMatchers ? {
                    match: getMiddlewareRouteMatcher(middlewareMatchers),
                    page: "/",
                    matchers: middlewareMatchers
                } : undefined;
                try {
                    var ref5;
                    // we serve a separate manifest with all pages for the client in
                    // dev mode so that we can match a page after a rewrite on the client
                    // before it has been built and is populated in the _buildManifest
                    const sortedRoutes = getSortedRoutes(routedPages);
                    if (!((ref5 = this.sortedRoutes) == null ? void 0 : ref5.every((val, idx)=>val === sortedRoutes[idx]))) {
                        // emit the change so clients fetch the update
                        this.hotReloader.send(undefined, {
                            devPagesManifest: true
                        });
                    }
                    this.sortedRoutes = sortedRoutes;
                    this.dynamicRoutes = this.sortedRoutes.filter(isDynamicRoute).map((page)=>({
                            page,
                            match: getRouteMatcher(getRouteRegex(page))
                        }));
                    this.router.setDynamicRoutes(this.dynamicRoutes);
                    this.router.setCatchallMiddleware(this.generateCatchAllMiddlewareRoute(true));
                    if (!resolved) {
                        resolve();
                        resolved = true;
                    }
                } catch (e) {
                    if (!resolved) {
                        reject(e);
                        resolved = true;
                    } else {
                        Log.warn("Failed to reload dynamic routes:", e);
                    }
                }
            });
        });
    }
    async stopWatcher() {
        if (!this.webpackWatcher) {
            return;
        }
        this.webpackWatcher.close();
        this.webpackWatcher = null;
    }
    async verifyTypeScript() {
        if (this.verifyingTypeScript) {
            return;
        }
        try {
            this.verifyingTypeScript = true;
            const verifyResult = await verifyTypeScriptSetup({
                dir: this.dir,
                intentDirs: [
                    this.pagesDir,
                    this.appDir
                ].filter(Boolean),
                typeCheckPreflight: false,
                tsconfigPath: this.nextConfig.typescript.tsconfigPath,
                disableStaticImages: this.nextConfig.images.disableStaticImages,
                isAppDirEnabled: !!this.appDir
            });
            if (verifyResult.version) {
                this.usingTypeScript = true;
            }
        } finally{
            this.verifyingTypeScript = false;
        }
    }
    async prepare() {
        setGlobal("distDir", this.distDir);
        setGlobal("phase", PHASE_DEVELOPMENT_SERVER);
        await this.verifyTypeScript();
        this.customRoutes = await loadCustomRoutes(this.nextConfig);
        // reload router
        const { redirects , rewrites , headers  } = this.customRoutes;
        if (rewrites.beforeFiles.length || rewrites.afterFiles.length || rewrites.fallback.length || redirects.length || headers.length) {
            this.router = new Router(this.generateRoutes());
        }
        this.hotReloader = new HotReloader(this.dir, {
            pagesDir: this.pagesDir,
            distDir: this.distDir,
            config: this.nextConfig,
            previewProps: this.getPreviewProps(),
            buildId: this.buildId,
            rewrites,
            appDir: this.appDir
        });
        await super.prepare();
        await this.addExportPathMapRoutes();
        await this.hotReloader.start(true);
        await this.startWatcher();
        this.setDevReady();
        if (this.nextConfig.experimental.nextScriptWorkers) {
            await verifyPartytownSetup(this.dir, pathJoin(this.distDir, CLIENT_STATIC_FILES_PATH));
        }
        const telemetry = new Telemetry({
            distDir: this.distDir
        });
        telemetry.record(eventCliSession(this.distDir, this.nextConfig, {
            webpackVersion: 5,
            cliCommand: "dev",
            isSrcDir: !!this.pagesDir && relative(this.dir, this.pagesDir).startsWith("src") || !!this.appDir && relative(this.dir, this.appDir).startsWith("src"),
            hasNowJson: !!await findUp("now.json", {
                cwd: this.dir
            }),
            isCustomServer: this.isCustomServer
        }));
        // This is required by the tracing subsystem.
        setGlobal("telemetry", telemetry);
        process.on("unhandledRejection", (reason)=>{
            this.logErrorWithOriginalStack(reason, "unhandledRejection").catch(()=>{});
        });
        process.on("uncaughtException", (err)=>{
            this.logErrorWithOriginalStack(err, "uncaughtException").catch(()=>{});
        });
    }
    async close() {
        await this.stopWatcher();
        await this.getStaticPathsWorker().end();
        if (this.hotReloader) {
            await this.hotReloader.stop();
        }
    }
    async hasPage(pathname) {
        let normalizedPath;
        try {
            normalizedPath = normalizePagePath(pathname);
        } catch (err) {
            console.error(err);
            // if normalizing the page fails it means it isn't valid
            // so it doesn't exist so don't throw and return false
            // to ensure we return 404 instead of 500
            return false;
        }
        if (isMiddlewareFile(normalizedPath)) {
            return findPageFile(this.dir, normalizedPath, this.nextConfig.pageExtensions, false).then(Boolean);
        }
        let appFile = null;
        let pagesFile = null;
        if (this.appDir) {
            appFile = await findPageFile(this.appDir, normalizedPath + "/page", this.nextConfig.pageExtensions, true);
        }
        if (this.pagesDir) {
            pagesFile = await findPageFile(this.pagesDir, normalizedPath, this.nextConfig.pageExtensions, false);
        }
        if (appFile && pagesFile) {
            throw new Error(`Conflicting app and page file found: "app${appFile}" and "pages${pagesFile}". Please remove one to continue.`);
        }
        return Boolean(appFile || pagesFile);
    }
    async _beforeCatchAllRender(req, res, params, parsedUrl) {
        const { pathname  } = parsedUrl;
        const pathParts = params.path || [];
        const path = `/${pathParts.join("/")}`;
        // check for a public file, throwing error if there's a
        // conflicting page
        let decodedPath;
        try {
            decodedPath = decodeURIComponent(path);
        } catch (_) {
            throw new DecodeError("failed to decode param");
        }
        if (await this.hasPublicFile(decodedPath)) {
            if (await this.hasPage(pathname)) {
                const err = new Error(`A conflicting public file and page file was found for path ${pathname} https://nextjs.org/docs/messages/conflicting-public-file-page`);
                res.statusCode = 500;
                await this.renderError(err, req, res, pathname, {});
                return true;
            }
            await this.servePublic(req, res, pathParts);
            return true;
        }
        return false;
    }
    setupWebSocketHandler(server, _req) {
        if (!this.addedUpgradeListener) {
            var ref17;
            this.addedUpgradeListener = true;
            server = server || ((ref17 = _req == null ? void 0 : _req.originalRequest.socket) == null ? void 0 : ref17.server);
            if (!server) {
                // this is very unlikely to happen but show an error in case
                // it does somehow
                Log.error(`Invalid IncomingMessage received, make sure http.createServer is being used to handle requests.`);
            } else {
                const { basePath  } = this.nextConfig;
                server.on("upgrade", (req, socket, head)=>{
                    var ref;
                    let assetPrefix = (this.nextConfig.assetPrefix || "").replace(/^\/+/, "");
                    // assetPrefix can be a proxy server with a url locally
                    // if so, it's needed to send these HMR requests with a rewritten url directly to /_next/webpack-hmr
                    // otherwise account for a path-like prefix when listening to socket events
                    if (assetPrefix.startsWith("http")) {
                        assetPrefix = "";
                    } else if (assetPrefix) {
                        assetPrefix = `/${assetPrefix}`;
                    }
                    if ((ref = req.url) == null ? void 0 : ref.startsWith(`${basePath || assetPrefix || ""}/_next/webpack-hmr`)) {
                        var ref16;
                        (ref16 = this.hotReloader) == null ? void 0 : ref16.onHMR(req, socket, head);
                    } else {
                        this.handleUpgrade(req, socket, head);
                    }
                });
            }
        }
    }
    async runMiddleware(params) {
        try {
            const result = await super.runMiddleware({
                ...params,
                onWarning: (warn)=>{
                    this.logErrorWithOriginalStack(warn, "warning");
                }
            });
            if ("finished" in result) {
                return result;
            }
            result.waitUntil.catch((error)=>{
                this.logErrorWithOriginalStack(error, "unhandledRejection");
            });
            return result;
        } catch (error) {
            if (error instanceof DecodeError) {
                throw error;
            }
            /**
       * We only log the error when it is not a MiddlewareNotFound error as
       * in that case we should be already displaying a compilation error
       * which is what makes the module not found.
       */ if (!(error instanceof MiddlewareNotFoundError)) {
                this.logErrorWithOriginalStack(error);
            }
            const err = getProperError(error);
            err.middleware = true;
            const { request , response , parsedUrl  } = params;
            /**
       * When there is a failure for an internal Next.js request from
       * middleware we bypass the error without finishing the request
       * so we can serve the required chunks to render the error.
       */ if (request.url.includes("/_next/static") || request.url.includes("/__nextjs_original-stack-frame")) {
                return {
                    finished: false
                };
            }
            response.statusCode = 500;
            this.renderError(err, request, response, parsedUrl.pathname);
            return {
                finished: true
            };
        }
    }
    async runEdgeFunction(params) {
        try {
            return super.runEdgeFunction({
                ...params,
                onWarning: (warn)=>{
                    this.logErrorWithOriginalStack(warn, "warning");
                }
            });
        } catch (error) {
            if (error instanceof DecodeError) {
                throw error;
            }
            this.logErrorWithOriginalStack(error, "warning");
            const err = getProperError(error);
            const { req , res , page  } = params;
            res.statusCode = 500;
            this.renderError(err, req, res, page);
            return null;
        }
    }
    async run(req, res, parsedUrl) {
        await this.devReady;
        this.setupWebSocketHandler(undefined, req);
        const { basePath  } = this.nextConfig;
        let originalPathname = null;
        if (basePath && pathHasPrefix(parsedUrl.pathname || "/", basePath)) {
            // strip basePath before handling dev bundles
            // If replace ends up replacing the full url it'll be `undefined`, meaning we have to default it to `/`
            originalPathname = parsedUrl.pathname;
            parsedUrl.pathname = removePathPrefix(parsedUrl.pathname || "/", basePath);
        }
        const { pathname  } = parsedUrl;
        if (pathname.startsWith("/_next")) {
            if (await fileExists(pathJoin(this.publicDir, "_next"))) {
                throw new Error(PUBLIC_DIR_MIDDLEWARE_CONFLICT);
            }
        }
        const { finished =false  } = await this.hotReloader.run(req.originalRequest, res.originalResponse, parsedUrl);
        if (finished) {
            return;
        }
        if (originalPathname) {
            // restore the path before continuing so that custom-routes can accurately determine
            // if they should match against the basePath or not
            parsedUrl.pathname = originalPathname;
        }
        try {
            return await super.run(req, res, parsedUrl);
        } catch (error) {
            const err = getProperError(error);
            this.logErrorWithOriginalStack(err).catch(()=>{});
            if (!res.sent) {
                res.statusCode = 500;
                try {
                    return await this.renderError(err, req, res, pathname, {
                        __NEXT_PAGE: isError(err) && err.page || pathname || ""
                    });
                } catch (internalErr) {
                    console.error(internalErr);
                    res.body("Internal Server Error").send();
                }
            }
        }
    }
    async logErrorWithOriginalStack(err, type) {
        let usedOriginalStack = false;
        if (isError(err) && err.stack) {
            try {
                const frames = parseStack(err.stack);
                const frame = frames.find(({ file  })=>{
                    return !(file == null ? void 0 : file.startsWith("eval")) && !(file == null ? void 0 : file.includes("web/adapter")) && !(file == null ? void 0 : file.includes("sandbox/context"));
                });
                if (frame.lineNumber && (frame == null ? void 0 : frame.file)) {
                    var ref, ref19, ref20, ref21, ref22, ref23;
                    const moduleId = frame.file.replace(/^(webpack-internal:\/\/\/|file:\/\/)/, "");
                    const src = getErrorSource(err);
                    const compilation = src === COMPILER_NAMES.edgeServer ? (ref = this.hotReloader) == null ? void 0 : (ref19 = ref.edgeServerStats) == null ? void 0 : ref19.compilation : (ref20 = this.hotReloader) == null ? void 0 : (ref21 = ref20.serverStats) == null ? void 0 : ref21.compilation;
                    const source = await getSourceById(!!((ref22 = frame.file) == null ? void 0 : ref22.startsWith(sep)) || !!((ref23 = frame.file) == null ? void 0 : ref23.startsWith("file:")), moduleId, compilation);
                    const originalFrame = await createOriginalStackFrame({
                        line: frame.lineNumber,
                        column: frame.column,
                        source,
                        frame,
                        modulePath: moduleId,
                        rootDirectory: this.dir,
                        errorMessage: err.message,
                        compilation
                    });
                    if (originalFrame) {
                        const { originalCodeFrame , originalStackFrame  } = originalFrame;
                        const { file , lineNumber , column , methodName  } = originalStackFrame;
                        Log[type === "warning" ? "warn" : "error"](`${file} (${lineNumber}:${column}) @ ${methodName}`);
                        if (src === COMPILER_NAMES.edgeServer) {
                            err = err.message;
                        }
                        if (type === "warning") {
                            Log.warn(err);
                        } else if (type) {
                            Log.error(`${type}:`, err);
                        } else {
                            Log.error(err);
                        }
                        console[type === "warning" ? "warn" : "error"](originalCodeFrame);
                        usedOriginalStack = true;
                    }
                }
            } catch (_) {
            // failed to load original stack using source maps
            // this un-actionable by users so we don't show the
            // internal error and only show the provided stack
            }
        }
        if (!usedOriginalStack) {
            if (type === "warning") {
                Log.warn(err);
            } else if (type) {
                Log.error(`${type}:`, err);
            } else {
                Log.error(err);
            }
        }
    }
    // override production loading of routes-manifest
    getCustomRoutes() {
        // actual routes will be loaded asynchronously during .prepare()
        return {
            redirects: [],
            rewrites: {
                beforeFiles: [],
                afterFiles: [],
                fallback: []
            },
            headers: []
        };
    }
    getPreviewProps() {
        if (this._devCachedPreviewProps) {
            return this._devCachedPreviewProps;
        }
        return this._devCachedPreviewProps = {
            previewModeId: crypto.randomBytes(16).toString("hex"),
            previewModeSigningKey: crypto.randomBytes(32).toString("hex"),
            previewModeEncryptionKey: crypto.randomBytes(32).toString("hex")
        };
    }
    getPagesManifest() {
        return undefined;
    }
    getAppPathsManifest() {
        return undefined;
    }
    getMiddleware() {
        return this.middleware;
    }
    getEdgeFunctions() {
        return this.edgeFunctions ?? [];
    }
    getServerComponentManifest() {
        return undefined;
    }
    getServerCSSManifest() {
        return undefined;
    }
    getFontLoaderManifest() {
        return undefined;
    }
    async hasMiddleware() {
        return this.hasPage(this.actualMiddlewareFile);
    }
    async ensureMiddleware() {
        return this.hotReloader.ensurePage({
            page: this.actualMiddlewareFile,
            clientOnly: false
        });
    }
    async ensureEdgeFunction({ page , appPaths  }) {
        return this.hotReloader.ensurePage({
            page,
            appPaths,
            clientOnly: false
        });
    }
    generateRoutes() {
        const { fsRoutes , ...otherRoutes } = super.generateRoutes();
        // In development we expose all compiled files for react-error-overlay's line show feature
        // We use unshift so that we're sure the routes is defined before Next's default routes
        fsRoutes.unshift({
            match: getPathMatch("/_next/development/:path*"),
            type: "route",
            name: "_next/development catchall",
            fn: async (req, res, params)=>{
                const p = pathJoin(this.distDir, ...params.path || []);
                await this.serveStatic(req, res, p);
                return {
                    finished: true
                };
            }
        });
        fsRoutes.unshift({
            match: getPathMatch(`/_next/${CLIENT_STATIC_FILES_PATH}/${this.buildId}/${DEV_CLIENT_PAGES_MANIFEST}`),
            type: "route",
            name: `_next/${CLIENT_STATIC_FILES_PATH}/${this.buildId}/${DEV_CLIENT_PAGES_MANIFEST}`,
            fn: async (_req, res)=>{
                res.statusCode = 200;
                res.setHeader("Content-Type", "application/json; charset=utf-8");
                res.body(JSON.stringify({
                    pages: this.sortedRoutes
                })).send();
                return {
                    finished: true
                };
            }
        });
        fsRoutes.unshift({
            match: getPathMatch(`/_next/${CLIENT_STATIC_FILES_PATH}/${this.buildId}/${DEV_MIDDLEWARE_MANIFEST}`),
            type: "route",
            name: `_next/${CLIENT_STATIC_FILES_PATH}/${this.buildId}/${DEV_MIDDLEWARE_MANIFEST}`,
            fn: async (_req, res)=>{
                var ref;
                res.statusCode = 200;
                res.setHeader("Content-Type", "application/json; charset=utf-8");
                res.body(JSON.stringify(((ref = this.getMiddleware()) == null ? void 0 : ref.matchers) ?? [])).send();
                return {
                    finished: true
                };
            }
        });
        fsRoutes.push({
            match: getPathMatch("/:path*"),
            type: "route",
            name: "catchall public directory route",
            fn: async (req, res, params, parsedUrl)=>{
                const { pathname  } = parsedUrl;
                if (!pathname) {
                    throw new Error("pathname is undefined");
                }
                // Used in development to check public directory paths
                if (await this._beforeCatchAllRender(req, res, params, parsedUrl)) {
                    return {
                        finished: true
                    };
                }
                return {
                    finished: false
                };
            }
        });
        return {
            fsRoutes,
            ...otherRoutes
        };
    }
    // In development public files are not added to the router but handled as a fallback instead
    generatePublicRoutes() {
        return [];
    }
    // In development dynamic routes cannot be known ahead of time
    getDynamicRoutes() {
        return [];
    }
    _filterAmpDevelopmentScript(html, event) {
        if (event.code !== "DISALLOWED_SCRIPT_TAG") {
            return true;
        }
        const snippetChunks = html.split("\n");
        let snippet;
        if (!(snippet = html.split("\n")[event.line - 1]) || !(snippet = snippet.substring(event.col))) {
            return true;
        }
        snippet = snippet + snippetChunks.slice(event.line).join("\n");
        snippet = snippet.substring(0, snippet.indexOf("</script>"));
        return !snippet.includes("data-amp-development-mode-only");
    }
    async getStaticPaths({ pathname , originalAppPath  }) {
        // we lazy load the staticPaths to prevent the user
        // from waiting on them for the page to load in dev mode
        const __getStaticPaths = async ()=>{
            const { configFileName , publicRuntimeConfig , serverRuntimeConfig , httpAgentOptions , experimental: { enableUndici  } ,  } = this.nextConfig;
            const { locales , defaultLocale  } = this.nextConfig.i18n || {};
            const pathsResult = await this.getStaticPathsWorker().loadStaticPaths({
                distDir: this.distDir,
                pathname,
                config: {
                    configFileName,
                    publicRuntimeConfig,
                    serverRuntimeConfig
                },
                httpAgentOptions,
                enableUndici,
                locales,
                defaultLocale,
                originalAppPath,
                isAppPath: !!originalAppPath
            });
            return pathsResult;
        };
        const { paths: staticPaths , fallback  } = (await withCoalescedInvoke(__getStaticPaths)(`staticPaths-${pathname}`, [])).value;
        return {
            staticPaths,
            fallbackMode: fallback === "blocking" ? "blocking" : fallback === true ? "static" : fallback
        };
    }
    async ensureApiPage(pathname) {
        return this.hotReloader.ensurePage({
            page: pathname,
            clientOnly: false
        });
    }
    async findPageComponents({ pathname , query , params , isAppPath , appPaths  }) {
        await this.devReady;
        const compilationErr = await this.getCompilationError(pathname);
        if (compilationErr) {
            // Wrap build errors so that they don't get logged again
            throw new WrappedBuildError(compilationErr);
        }
        try {
            await this.hotReloader.ensurePage({
                page: pathname,
                appPaths,
                clientOnly: false
            });
            // When the new page is compiled, we need to reload the server component
            // manifest.
            if (!!this.appDir) {
                this.serverComponentManifest = super.getServerComponentManifest();
                this.serverCSSManifest = super.getServerCSSManifest();
            }
            this.fontLoaderManifest = super.getFontLoaderManifest();
            return super.findPageComponents({
                pathname,
                query,
                params,
                isAppPath
            });
        } catch (err) {
            if (err.code !== "ENOENT") {
                throw err;
            }
            return null;
        }
    }
    async getFallbackErrorComponents() {
        await this.hotReloader.buildFallbackError();
        // Build the error page to ensure the fallback is built too.
        // TODO: See if this can be moved into hotReloader or removed.
        await this.hotReloader.ensurePage({
            page: "/_error",
            clientOnly: false
        });
        return await loadDefaultErrorComponents(this.distDir);
    }
    setImmutableAssetCacheControl(res) {
        res.setHeader("Cache-Control", "no-store, must-revalidate");
    }
    servePublic(req, res, pathParts) {
        const p = pathJoin(this.publicDir, ...pathParts);
        return this.serveStatic(req, res, p);
    }
    async hasPublicFile(path) {
        try {
            const info = await fs.promises.stat(pathJoin(this.publicDir, path));
            return info.isFile();
        } catch (_) {
            return false;
        }
    }
    async getCompilationError(page) {
        const errors = await this.hotReloader.getCompilationErrors(page);
        if (errors.length === 0) return;
        // Return the very first error we found.
        return errors[0];
    }
    isServeableUrl(untrustedFileUrl) {
        // This method mimics what the version of `send` we use does:
        // 1. decodeURIComponent:
        //    https://github.com/pillarjs/send/blob/0.17.1/index.js#L989
        //    https://github.com/pillarjs/send/blob/0.17.1/index.js#L518-L522
        // 2. resolve:
        //    https://github.com/pillarjs/send/blob/de073ed3237ade9ff71c61673a34474b30e5d45b/index.js#L561
        let decodedUntrustedFilePath;
        try {
            // (1) Decode the URL so we have the proper file name
            decodedUntrustedFilePath = decodeURIComponent(untrustedFileUrl);
        } catch  {
            return false;
        }
        // (2) Resolve "up paths" to determine real request
        const untrustedFilePath = pathResolve(decodedUntrustedFilePath);
        // don't allow null bytes anywhere in the file path
        if (untrustedFilePath.indexOf("\0") !== -1) {
            return false;
        }
        // During development mode, files can be added while the server is running.
        // Checks for .next/static, .next/server, static and public.
        // Note that in development .next/server is available for error reporting purposes.
        // see `packages/next/server/next-server.ts` for more details.
        if (untrustedFilePath.startsWith(pathJoin(this.distDir, "static") + sep) || untrustedFilePath.startsWith(pathJoin(this.distDir, "server") + sep) || untrustedFilePath.startsWith(pathJoin(this.dir, "static") + sep) || untrustedFilePath.startsWith(pathJoin(this.dir, "public") + sep)) {
            return true;
        }
        return false;
    }
};

//# sourceMappingURL=next-dev-server.js.map